<?php
namespace App\Controllers\Api\Ipn;

use App\Controllers\BaseController;
use App\Libraries\Payment;
use CodeIgniter\HTTP\ResponseInterface;
use App\Libraries\RazorpayClient;

class Razorpay extends BaseController
{
    private $razorpay;

    public function __construct()
    {
        $this->razorpay = new RazorpayClient();
    }

    public function index(): ResponseInterface
    {
        $bodyRaw = file_get_contents('php://input');
        $signature = $this->request->getHeaderLine("X-Razorpay-Signature");

        if (empty($signature)) {
            return $this->respond(["status" => "fail", "message" => "Missing signature"], 400);
        }

        $result = $this->razorpay->process_webhook($bodyRaw, $signature);

        log_message("error", "Razorpay IPN: " . json_encode($result));

        if (!$result["event"]) {
            // Webhook not processed, skip
            return $this->respond(["status" => "ok"], 200);
        }

        $customValues = explode("_", $result["custom_id"]);

        $app_id = $customValues[0];
        $user_id = $customValues[1];
        $plan_id = $customValues[2];

        $payment = new Payment([
            "user_id" => $user_id,
            "app_id" => $app_id,
            "plan_id" => $plan_id,
        ]);

        switch ($result["event_type"]) {
            case 'subscription.charged':
                if ($result['paid_count'] == 1) {
                    $payment->create_subscription([
                        "subscribe_external_id" => $result["order_id"],
                        "customer_external_id" => $result["customer_id"],
                        "price" => (float)$result["amount"],
                        "method_route" => "razorpay",
                    ]);
                }
                
                $payment->create_transaction([
                    "subscribe_external_id" => $result["order_id"],
                    "external_uid" => $result["resource_id"],
                    "price" => (float)$result["amount"],
                    "method_route" => "razorpay",
                    "event_data" => $result["event_data"],
                ], $result['paid_count'] == 1 ? false : true);
                break;
                
            case 'subscription.cancelled':
                $payment->cancel_subscription([
                    "cancel_at" => time(),
                ]);
                break;
                
            case 'payment_link.paid':
                $payment->create_subscription([
                    "subscribe_external_id" => $result["order_id"],
                    "customer_external_id" => $result["customer_id"],
                    "price" => (float)$result["amount"],
                    "method_route" => "razorpay",
                ]);
                
                $payment->create_transaction([
                    "subscribe_external_id" => $result["order_id"],
                    "external_uid" => $result["resource_id"],
                    "price" => (float)$result["amount"],
                    "method_route" => "razorpay",
                    "event_data" => $result["event_data"],
                ]);
                break;

            default:
                return $this->respond(["message" => "Unhandled event type"], 400);
        }

        return $this->respond(["status" => "ok"], 200);
    }
}